package application.model;

import java.io.File;
import java.io.FileNotFoundException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;

import org.bouncycastle.openpgp.PGPException;
import org.bouncycastle.openpgp.PGPPublicKey;
import org.bouncycastle.util.encoders.Hex;

import exceptions.LibsignerInvalidPrivKeyException;
import exceptions.LibsignerInvalidPubKeyException;
import exceptions.LibsignerInvalidSecKeyException;
import exceptions.LibsignerReadFailedException;
import javafx.collections.FXCollections;
import javafx.collections.ObservableList;
import javafx.concurrent.Task;
import libsigner.Utils;

public class GPGState {
	private PGPPublicKey pubKey;
	private String expiryDate;
	private String fingerprint;
	private String userID;
	private boolean keysCreated;
	private boolean is_pending_sign;
	private boolean is_pending_decompression;
	private ObservableList<TableModel> tablemodel;
	
	public GPGState(){
		tablemodel = FXCollections.observableArrayList();

		is_pending_sign = false;
		is_pending_decompression = false;

		expiryDate="00/00/0000";
		fingerprint = "";
		userID = "";

		updateKeyInfo();

	}

	public void loadPubKey(){
		if(keysCreated){
			try {
				pubKey = Utils.retrievePubKeyFromPubring("pubring.gpg");
			} catch (LibsignerReadFailedException | FileNotFoundException | LibsignerInvalidPubKeyException
					| PGPException e) {
				e.printStackTrace();
			}
		}
	}

	public void updateKeyInfo(){
		checkIfExists();

		loadPubKey();

		if(keysCreated){
			setFingerprint();
			setExpiryDate();
			setUserID();
		}
	}
	
	public void checkIfExists(){
		boolean pubKeyExists = new File("pubring.gpg").exists();
		boolean privKeyExists = new File("secring.gpg").exists();

		if(pubKeyExists && privKeyExists)
			keysCreated = true;
		else
			keysCreated = false;
	}

	private void setFingerprint(){
		String temp = Hex.toHexString( pubKey.getFingerprint() ).toUpperCase();
		fingerprint = temp.substring(0, 4) + " " +
				temp.substring(4, 8) + " " +
				temp.substring(8, 12) + " " +
				temp.substring(12, 16) + " " +
				temp.substring(16, 20) + " " +
				temp.substring(20, 24) + " " +
				temp.substring(24, 28) + " " +
				temp.substring(28, 32) + " " +
				temp.substring(32, 36) + " " +
				temp.substring(36, 40) + " ";
	}

	private void setExpiryDate(){
		//Initializing the expiry date
		Calendar cal = Calendar.getInstance();
		cal.setTime(pubKey.getCreationTime());
		cal.roll(Calendar.YEAR, 2);
		expiryDate = (new SimpleDateFormat("dd/MM/yyyy HH:mm")).format(cal.getTime());
	}

	private void setUserID(){
		Iterator<?> it = pubKey.getUserIDs();
		while(it.hasNext()){
			userID = (String) it.next();
		}
	}

	public Task<HashMap<File, Object>> decompress(){
		List<File> pending_decompression = new ArrayList<File>();
		for (Iterator<TableModel> iterator = tablemodel.iterator(); iterator.hasNext();) {
			TableModel tableModel2 = (TableModel) iterator.next();

			if(tableModel2.getArquivo_acao().equals("Abrir")){
				pending_decompression.add(tableModel2.getArquivo_file_object());
			}
		}

		Task<HashMap<File, Object>> task = new Task<HashMap<File, Object>>(){

			@Override
			protected HashMap<File, Object> call() throws LibsignerReadFailedException, FileNotFoundException, LibsignerInvalidPubKeyException, PGPException{
				HashMap<File, Object> result = Utils.batchDecompressVerify(pending_decompression, "pubring.gpg");

				return result;
			}

		};
		
		return task;
	}

	public Task<HashMap<File, Exception>> sign(char[] password){
		List<File> pending_sign = new ArrayList<File>();
		for (Iterator<TableModel> iterator = tablemodel.iterator(); iterator.hasNext();) {
			TableModel tableModel2 = (TableModel) iterator.next();

			if(tableModel2.getArquivo_acao().equals("Assinar")){
				pending_sign.add(tableModel2.getArquivo_file_object());
			}
		}

		Task<HashMap<File,Exception>> task = new Task<HashMap<File,Exception>>(){

			@Override
			protected HashMap<File,Exception> call() throws LibsignerReadFailedException, LibsignerInvalidPrivKeyException, FileNotFoundException, LibsignerInvalidSecKeyException, PGPException{
				HashMap<File, Exception> result = Utils.batchSign(pending_sign, "secring.gpg", password);

				return result;
			}

		};
		
		return task;
	}

	public boolean isPendingEmpty(){
		return is_pending_sign;
	}

	public boolean isDecompressionEmpty(){
		return is_pending_decompression;
	}

	public String getUserID(){
		return userID;
	}

	public String getFingerprint(){
		return fingerprint;
	}

	public String getExpiryDate() {
		return expiryDate;
	}

	public boolean getKeysCreated(){
		updateKeyInfo();
		return keysCreated;
	}

	public ObservableList<TableModel> getTableModel(){
		return tablemodel;
	}

	public void add_to_sign(File file){
		tablemodel.add(
				new TableModel(file.getName(), file.getAbsolutePath(), new String("Assinar"), file )
				);

		is_pending_sign = true;
	}

	public void add_to_decompress(File file){
		tablemodel.add(
				new TableModel(file.getName(), file.getAbsolutePath(), new String("Abrir"), file )
				);

		is_pending_decompression = true;
	}
}
