package libsigner;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.math.BigInteger;
import java.nio.file.FileSystems;
import java.nio.file.Files;
import java.nio.file.Paths;
import java.security.SecureRandom;
import java.util.Calendar;
import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;

import org.bouncycastle.bcpg.ArmoredOutputStream;
import org.bouncycastle.bcpg.BCPGOutputStream;
import org.bouncycastle.bcpg.HashAlgorithmTags;
import org.bouncycastle.bcpg.SymmetricKeyAlgorithmTags;
import org.bouncycastle.bcpg.sig.Features;
import org.bouncycastle.bcpg.sig.KeyFlags;
import org.bouncycastle.crypto.generators.RSAKeyPairGenerator;
import org.bouncycastle.crypto.params.RSAKeyGenerationParameters;
import org.bouncycastle.openpgp.PGPCompressedData;
import org.bouncycastle.openpgp.PGPCompressedDataGenerator;
import org.bouncycastle.openpgp.PGPEncryptedData;
import org.bouncycastle.openpgp.PGPException;
import org.bouncycastle.openpgp.PGPKeyPair;
import org.bouncycastle.openpgp.PGPKeyRingGenerator;
import org.bouncycastle.openpgp.PGPLiteralData;
import org.bouncycastle.openpgp.PGPLiteralDataGenerator;
import org.bouncycastle.openpgp.PGPObjectFactory;
import org.bouncycastle.openpgp.PGPOnePassSignature;
import org.bouncycastle.openpgp.PGPOnePassSignatureList;
import org.bouncycastle.openpgp.PGPPrivateKey;
import org.bouncycastle.openpgp.PGPPublicKey;
import org.bouncycastle.openpgp.PGPPublicKeyRing;
import org.bouncycastle.openpgp.PGPPublicKeyRingCollection;
import org.bouncycastle.openpgp.PGPSecretKey;
import org.bouncycastle.openpgp.PGPSecretKeyRing;
import org.bouncycastle.openpgp.PGPSecretKeyRingCollection;
import org.bouncycastle.openpgp.PGPSignature;
import org.bouncycastle.openpgp.PGPSignatureGenerator;
import org.bouncycastle.openpgp.PGPSignatureList;
import org.bouncycastle.openpgp.PGPSignatureSubpacketGenerator;
import org.bouncycastle.openpgp.PGPUtil;
import org.bouncycastle.openpgp.operator.PBESecretKeyEncryptor;
import org.bouncycastle.openpgp.operator.PGPDigestCalculator;
import org.bouncycastle.openpgp.operator.bc.BcPBESecretKeyDecryptorBuilder;
import org.bouncycastle.openpgp.operator.bc.BcPBESecretKeyEncryptorBuilder;
import org.bouncycastle.openpgp.operator.bc.BcPGPContentSignerBuilder;
import org.bouncycastle.openpgp.operator.bc.BcPGPDigestCalculatorProvider;
import org.bouncycastle.openpgp.operator.bc.BcPGPKeyPair;
import org.bouncycastle.openpgp.operator.jcajce.JcaKeyFingerprintCalculator;
import org.bouncycastle.openpgp.operator.jcajce.JcaPGPContentVerifierBuilderProvider;
import org.bouncycastle.util.io.Streams;

import exceptions.LibsignerFileException;
import exceptions.LibsignerInvalidPrivKeyException;
import exceptions.LibsignerInvalidPubKeyException;
import exceptions.LibsignerInvalidSecKeyException;
import exceptions.LibsignerReadFailedException;
import exceptions.LibsignerWriteFailedExeception;

public class Utils {
	public static void generateRSAPair(String id, char[] password, String pathKeyPub, String pathKeyPriv) throws PGPException, LibsignerWriteFailedExeception {
		int s2kcount = 0xc0;
		
		//Generator of a RSA keypair
		RSAKeyPairGenerator  kpg = new RSAKeyPairGenerator();
		
		//Initialize key parameters
		//	4096 -> lenght
		//	128 -> certainty (see https://www.keylength.com/en/4/)
        kpg.init
        (new RSAKeyGenerationParameters
         (BigInteger.valueOf(0x10001),
          new SecureRandom(), 4096, 128));
		
        //Creates the sign key
        PGPKeyPair rsakp_sign =
            new BcPGPKeyPair
            (PGPPublicKey.RSA_SIGN, kpg.generateKeyPair(), new Date());

        //Creates the encryption key
        PGPKeyPair rsakp_enc =
            new BcPGPKeyPair
            (PGPPublicKey.RSA_ENCRYPT, kpg.generateKeyPair(), new Date());
        
        //Creates the subpacket generator for the public key
        PGPSignatureSubpacketGenerator signhashgen = new PGPSignatureSubpacketGenerator();
        
        //Add self-signature metadata on the sign key
        // 1) Key can be used to sign and certify other keys
        signhashgen.setKeyFlags
            (false, KeyFlags.SIGN_DATA|KeyFlags.CERTIFY_OTHER);

        // 2) Crypto algorithm preferences 
        signhashgen.setPreferredSymmetricAlgorithms
            (false, new int[] {
                SymmetricKeyAlgorithmTags.AES_256,
                SymmetricKeyAlgorithmTags.AES_192,
                SymmetricKeyAlgorithmTags.AES_128
            });
        signhashgen.setPreferredHashAlgorithms
            (false, new int[] {
                HashAlgorithmTags.SHA256,
                //HashAlgorithmTags.SHA1,
                HashAlgorithmTags.SHA384,
                HashAlgorithmTags.SHA512,
                HashAlgorithmTags.SHA224,
            });

        // 3) Add the feature of modification detection
        signhashgen.setFeature
            (false, Features.FEATURE_MODIFICATION_DETECTION);
        
        // 4) Calculates expiration date
        Calendar current = Calendar.getInstance();
        Calendar expiry = Calendar.getInstance();
        expiry.roll(Calendar.YEAR, 2);
        
        long secondsToExpire = (expiry.getTime().getTime() - current.getTime().getTime())/1000;
        
        //Set the expiration time in the key
        signhashgen.setKeyExpirationTime(false, secondsToExpire);
        
        
        //Creates the subpacket generator for the encryption key
        PGPSignatureSubpacketGenerator enchashgen = new PGPSignatureSubpacketGenerator();
        
        //Add metadata to the encryption key
        enchashgen.setKeyFlags
            (false, KeyFlags.ENCRYPT_COMMS|KeyFlags.ENCRYPT_STORAGE);
        
        //Creates the signature calculator for keyring creation
        PGPDigestCalculator sha1Calc = new BcPGPDigestCalculatorProvider().get(HashAlgorithmTags.SHA1);
        PGPDigestCalculator sha256Calc = new BcPGPDigestCalculatorProvider().get(HashAlgorithmTags.SHA256);
        
        //Builds the private key
        PBESecretKeyEncryptor pske = (new BcPBESecretKeyEncryptorBuilder
             (PGPEncryptedData.AES_256, sha256Calc, s2kcount))
            .build(password);
        
        //Generates the keyring with the sign key
        PGPKeyRingGenerator keyRingGen =
            new PGPKeyRingGenerator
            (PGPSignature.POSITIVE_CERTIFICATION, rsakp_sign,
             id, sha1Calc, signhashgen.generate(), null,
             new BcPGPContentSignerBuilder
             (rsakp_sign.getPublicKey().getAlgorithm(),
              HashAlgorithmTags.SHA1),
             pske);

        //Adds the encryption key to keyring
        keyRingGen.addSubKey
            (rsakp_enc, enchashgen.generate(), null);

        //Generate public key ring, dump to file.
        PGPPublicKeyRing pkr = keyRingGen.generatePublicKeyRing();
		try (BufferedOutputStream pubout = new BufferedOutputStream(new FileOutputStream(pathKeyPub))){
			pkr.encode(pubout);
		} catch (IOException e) {
			throw new LibsignerWriteFailedExeception(e);
		}

        //Generate private key, dump to file.
        PGPSecretKeyRing skr = keyRingGen.generateSecretKeyRing();
        try (BufferedOutputStream secout = new BufferedOutputStream(new FileOutputStream(pathKeyPriv))){
        	skr.encode(secout);
            secout.close();
        }catch (IOException e) {
        	throw new LibsignerWriteFailedExeception(e);
		}


        password = null;
	}

	public static HashMap<File, Exception> batchSign(List<File> files, String pathKeyPriv, char[] password) throws LibsignerInvalidPrivKeyException, FileNotFoundException, LibsignerInvalidSecKeyException, PGPException, LibsignerReadFailedException {
		PGPSecretKey secKey = retrieveSecretKeyFromSecring(pathKeyPriv);
		PGPPrivateKey privKey;
		try{
			privKey = secKey.extractPrivateKey(new BcPBESecretKeyDecryptorBuilder(new BcPGPDigestCalculatorProvider()).build(password));
		}catch(PGPException e){
			throw new LibsignerInvalidPrivKeyException(e);
		}
		password = null;

		Iterator<File> it = files.iterator();
		HashMap<File, Exception> success = new HashMap<File, Exception>();
		while (it.hasNext()) {
			 File file = it.next();
			 try{
				 signFile(file.getPath(), secKey, privKey);
			 }catch(LibsignerFileException e){
				 success.put(file, e);
			 }
		}

		secKey = null;
		privKey = null;
		return success;
	}

	public static void signFile(String filepath, String pathKeyPriv, char[] password) throws FileNotFoundException, LibsignerInvalidSecKeyException, LibsignerInvalidPrivKeyException, PGPException, LibsignerFileException {
		PGPSecretKey secKey = retrieveSecretKeyFromSecring(pathKeyPriv);
		PGPPrivateKey privKey;

		try{
			privKey = secKey.extractPrivateKey(new BcPBESecretKeyDecryptorBuilder(new BcPGPDigestCalculatorProvider()).build(password));
		}catch(PGPException e){
			throw new LibsignerInvalidPrivKeyException(e);
		}

		password = null;
		signFile(filepath, secKey, privKey);
	}

	public static void signFile(String filepath, PGPSecretKey secKey, PGPPrivateKey privKey) throws FileNotFoundException, LibsignerFileException, PGPException{
        OutputStream out = new ArmoredOutputStream(new BufferedOutputStream(new FileOutputStream(filepath.replace(".pdf", ".sig"))));
        
        PGPSignatureGenerator sGen = new PGPSignatureGenerator(new BcPGPContentSignerBuilder(secKey.getPublicKey().getAlgorithm(), PGPUtil.SHA256));
        PGPSignatureSubpacketGenerator  spGen = new PGPSignatureSubpacketGenerator();
        
        sGen.init(PGPSignature.BINARY_DOCUMENT, privKey);

		Iterator<?> userIDs = secKey.getPublicKey().getUserIDs();
        spGen.setSignerUserID(false, (String)userIDs.next());
        sGen.setHashedSubpackets(spGen.generate());
        
        try{
	        PGPCompressedDataGenerator  compressDataGenerator = new PGPCompressedDataGenerator(PGPCompressedData.ZLIB);
	        BCPGOutputStream bcOutputStream = new BCPGOutputStream(compressDataGenerator.open(out));
	        sGen.generateOnePassVersion(false).encode(bcOutputStream);
	 
	        PGPLiteralDataGenerator literalDataGenerator = new PGPLiteralDataGenerator();
	        OutputStream literalDataGenOutputStream = literalDataGenerator.open(bcOutputStream, PGPLiteralData.BINARY, new File(filepath));
	        FileInputStream fis = new FileInputStream(filepath);
	        
	        int ch;
		
	        while ((ch = fis.read()) >= 0) {
	            literalDataGenOutputStream.write(ch);
	            sGen.update((byte)ch);
	        }
	 
	        literalDataGenerator.close();
	        fis.close();
	 
	        sGen.generate().encode(bcOutputStream);
	        compressDataGenerator.close();
	        out.close();
        }catch(IOException e){
        	throw new LibsignerFileException(e);
        }
	}

	public static HashMap<File, Object> batchDecompressVerify(List<File> files, String pathKeyPub) throws LibsignerReadFailedException, FileNotFoundException, LibsignerInvalidPubKeyException, PGPException{
		PGPPublicKey pubkey = retrievePubKeyFromPubring(pathKeyPub);

		Iterator<File> it = files.iterator();
		HashMap<File, Object> success = new HashMap<File, Object>();
		while (it.hasNext()) {
			 File file = it.next();
			 try {
				boolean ret = decompressAndVerify(file.getPath(), pubkey);
				success.put(file, ret);
			} catch (LibsignerWriteFailedExeception e) {
				success.put(file, e);
			}
		}

		return success;
	}

	public static boolean decompressAndVerify(String filepath, String pathKeyPub) throws IOException, PGPException, LibsignerReadFailedException, LibsignerInvalidPubKeyException, LibsignerWriteFailedExeception{
		return decompressAndVerify(filepath, retrievePubKeyFromPubring(pathKeyPub));
	}

	public static boolean decompressAndVerify(String filepath, PGPPublicKey pubkey) throws LibsignerReadFailedException, LibsignerWriteFailedExeception, PGPException {
		//code heavily based on http://stackoverflow.com/questions/19173181/bouncycastle-pgp-decrypt-and-verify
		InputStream input;
		
		try{
			input = new BufferedInputStream(new FileInputStream(filepath));
			
			input = PGPUtil.getDecoderStream(input);
		}catch(IOException e){
			throw new LibsignerReadFailedException(e);
		}
		
		PGPObjectFactory pgpF = new PGPObjectFactory(input, new JcaKeyFingerprintCalculator());
		
		ByteArrayOutputStream actualOutput = new ByteArrayOutputStream();
		PGPOnePassSignatureList onePassSignatureList = null;
		PGPSignatureList signatureList = null;
		PGPCompressedData compressedData;
		String originalFileName = null;
		Object message;
		
		try{
			message = pgpF.nextObject();
	
			while (message != null) {
				if (message instanceof PGPCompressedData) {
					compressedData = (PGPCompressedData) message;
					pgpF = new PGPObjectFactory(compressedData.getDataStream(), new JcaKeyFingerprintCalculator());
					message = pgpF.nextObject();
				}
			
				if (message instanceof PGPLiteralData) {
					Streams.pipeAll(((PGPLiteralData) message).getInputStream(), actualOutput);
					originalFileName = ((PGPLiteralData) message).getFileName();
				} else if (message instanceof PGPOnePassSignatureList) {
					onePassSignatureList = (PGPOnePassSignatureList) message;
				} else if (message instanceof PGPSignatureList) {
					signatureList = (PGPSignatureList) message;
				} else {
					throw new PGPException("Unknown message type");
				}
					message = pgpF.nextObject();
			}
			
			actualOutput.close();
			input.close();
		} catch(IOException e){
			throw new LibsignerReadFailedException(e);
		}
		
	    byte[] output = actualOutput.toByteArray();
	    
	    try{
	    	Files.write(Paths.get(Paths.get(filepath).getParent().toString() + FileSystems.getDefault().getSeparator() + originalFileName), output);
	    }catch(IOException e){
	    	throw new LibsignerWriteFailedExeception(e);
	    }

	    return verify(output, onePassSignatureList, signatureList, pubkey);
	}

	private static boolean verify(byte[] output, PGPOnePassSignatureList onePassSignatureList, PGPSignatureList signatureList,PGPPublicKey pubkey) throws PGPException {
		PGPOnePassSignature ops = onePassSignatureList.get(0);
	    ops.init(new JcaPGPContentVerifierBuilderProvider(), pubkey);
        ops.update(output);

        PGPSignature signature = signatureList.get(0);

        if(ops.verify(signature)){
        	return true;
		} else {
			return false;
		}
	}

	public static PGPPublicKey retrievePubKeyFromPubring(String pathKeyPub) throws FileNotFoundException, LibsignerReadFailedException, LibsignerInvalidPubKeyException, PGPException{
		InputStream KeyIn = new BufferedInputStream(new FileInputStream(pathKeyPub));
		PGPPublicKeyRingCollection pgpPub;

		try(InputStream input = PGPUtil.getDecoderStream(KeyIn)){
			pgpPub = new PGPPublicKeyRingCollection(input, new JcaKeyFingerprintCalculator());
		}catch(IOException e){
			throw new LibsignerReadFailedException(e);
		}catch(PGPException e){
			throw new LibsignerInvalidPubKeyException(e);
		}

		Iterator<PGPPublicKeyRing> keyRingIter = pgpPub.getKeyRings();
		while(keyRingIter.hasNext()){
			PGPPublicKeyRing keyRing = (PGPPublicKeyRing)keyRingIter.next();

			Iterator<PGPPublicKey> keyIter = keyRing.getPublicKeys();
			while(keyIter.hasNext()){
				PGPPublicKey key = (PGPPublicKey)keyIter.next();

				return key;
			}
		}

		throw new PGPException("No valid public key found");
	}

	public static PGPSecretKey retrieveSecretKeyFromSecring(String pathKeyPriv) throws FileNotFoundException, LibsignerReadFailedException, LibsignerInvalidSecKeyException, PGPException{
        InputStream keyIn = new BufferedInputStream(new FileInputStream(pathKeyPriv));
        PGPSecretKeyRingCollection pgpSec;

        try(InputStream input = PGPUtil.getDecoderStream(keyIn)){
        	pgpSec = new PGPSecretKeyRingCollection(input, new JcaKeyFingerprintCalculator());
        } catch (IOException e) {
			throw new LibsignerReadFailedException(e);
		} catch (PGPException e) {
			throw new LibsignerInvalidSecKeyException(e);
		}


        Iterator<PGPSecretKeyRing> keyRingIter = pgpSec.getKeyRings();
        while (keyRingIter.hasNext())
        {
            PGPSecretKeyRing keyRing = (PGPSecretKeyRing)keyRingIter.next();

            Iterator<PGPSecretKey> keyIter = keyRing.getSecretKeys();
            while (keyIter.hasNext())
            {
                PGPSecretKey key = (PGPSecretKey)keyIter.next();

                if (key.isSigningKey())
                {
                	return key;
                }
            }
        }

        throw new PGPException("No valid key found");
	}
	
}